unit Model3ds;

(* Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the gl3ds main unit.
 *
 * The Initial Developer of the Original Code is
 * Noeska Software.
 * Portions created by the Initial Developer are Copyright (C) 2002-2004
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 *  M van der Honing
 *  Sascha Willems
 *  Jan Michalowsky
 *
 *)

interface

uses classes, Model, glmath;

type

T3dsModel = Class(TBaseModel)
  private
    procedure MoveToPivot;
  public
    procedure LoadFromFile(AFileName: string); override;
    procedure LoadFromStream(Stream: TStream); override;
End;

implementation

uses SysUtils, Mesh, Material, windows;

const
  OVersion = 3; //expects 3ds version 3 files

  //Chunk ID's
  //  INT_PERCENTAGE = $0030; //hm transparency always has this subchunck
  MAIN3DS = $4D4D;
  VERS3DS = $0002;
  EDIT3DS = $3D3D;
  KEYF3DS = $B000;

  EDIT_OBJECT = $4000;

  MASTER_SCALE = $0100;

  OBJ_HIDDEN = $4010;
  OBJ_VERINFO = $3D3E;
  OBJ_TRIMESH = $4100;
  OBJ_LIGHT = $4600;
  OBJ_CAMERA = $4700;

  TRI_VERTEXL = $4110;
  TRI_FACEL1 = $4120;
  TRI_MATERIAL = $4130;
  TRI_MAPPINGCOORDS = $4140;
  TRI_MATRIX = $4160; //TRI_LOCAL; //Gives a matrix for each mesh?
  TRI_VISIBLE = $4165; //Is mesh visible or not


  MAT_MATRIAL = $AFFF;
  MAT_MATNAME = $A000;
  MAT_AMBIENT = $A010;
  MAT_DIFFUSE = $A020;
  MAT_SPECULAR = $A030;
  MAT_TRANSPARENCY = $A050; // Transparency Material
  MAT_TEXTURE = $A200;  //texmap
  MAT_OPACMAP = $A210;
  MAT_BUMPMAP = $A230;
  MAT_MAPFILE = $A300;
  MAT_VSCALE = $A354;
  MAT_USCALE = $A356;
  MAT_VOFF = $A35A;
  MAT_UOFF = $A358;
  MAT_TEXROT = $A35C;
  MAT_COLOR = $0010;
  MAT_COLOR24 = $0011;
  MAT_TWO_SIDE = $A081; //thanks sos

  KEYF_OBJDES = $B002;
  KEYF_OBJHIERARCH = $B010;
  KEYF_OBJPIVOT = $B013;

type
 TChunkHdr = packed record
    chunkID: Word;
    chunkLE: LongWord;
  end;

   T3DSColor = packed record
    r: Single;
    g: Single;
    b: Single;
  end;

   rgb = packed record
    r: Byte;
    g: Byte;
    b: Byte;
  end;

procedure T3dsModel.MoveToPivot;
var
  f, m: Integer;
  pivot: t3dpoint;
  tempPoint: t3dPoint;
begin
  //TODO: Should this be left out: e.g. implement it as gltranslatef at rendering?
  for m := 0 to FNumMeshes - 1 do
  begin
    if FMesh[m].NumVertex > 0 then
    begin
      f := 0;
      while f < FMesh[m].NumVertex do // go through all vertexes and
      begin
        pivot.x := FMesh[m].Pivot.x;
        pivot.y := FMesh[m].Pivot.y;
        pivot.z := FMesh[m].Pivot.z;

        tempPoint := FMesh[m].Vertex[f];

        tempPoint.x := tempPoint.x - pivot.x;
        tempPoint.y := tempPoint.y - pivot.y;
        tempPoint.z := tempPoint.z - pivot.z;

        FMesh[m].Vertex[f] := tempPoint;

        f := f + 1;
      end;
    end;
  end;
end;

procedure T3dsModel.LoadFromFile(AFileName: string);
var
  stream: TFilestream;
begin
  FPath := ExtractFilePath(AFilename);
  if FTexturePath = '' then FTexturePath:=FPath;

  stream := TFilestream.Create(AFilename, $0000);
  LoadFromStream(stream);
  stream.Free;
end;

procedure T3dsModel.LoadFromStream(stream: Tstream);
var
  buffer: TChunkHdr;
  chrbuf: Byte;
  color: t3dscolor;
  rgbcolor: rgb;
  Count, i, dummy: Word;
  tv: T3dPoint;
  tm: TMap;
  matarr: Byte;
  transp: short;
  bump, opac: short;
  tempscale: Single;
  matcount, matcountloop, mati, keyfcount: Word;
  m: LongWord;
  bumpmap, opacmap: Boolean;
  acount: LongWord;
  mcount: LongWord;
  mapname: string;
  singlebuffer: single;
  stringbuffer: string;
  wordbuffer: word;
  //singlebuffer: singe;
  doublebuffer: double;
  tempMap: TMap;
  //tempFace: TFace;
  tempbool: boolean;

  temp_diff: word;
begin
  //init hchunk
  FName := '';
  FMesh := nil;
  FMaterial := nil;
  FNumMeshes := 0; // set number of meshes to zero
  FNumMaterials := 0; // set number of materials to zero
  fmasterscale := 1;

  acount := 0;
  mcount := 0;
  keyfcount := 0;
  matarr := 0;
  bumpmap := false;
  opacmap := false;

  //start reading chunks
  //now all is read in once... This helps for some none standard 3ds files like those generated by cinema4d.
  while stream.Position < stream.Size do
  begin
    stream.Read(buffer, sizeof(TChunkHdr));
    //log.writeln('$'+IntToHex(buffer.chunkid,4));
    case buffer.chunkID of
      MAIN3DS:
      begin
        //No nothing seems to be needed here...
      end;
      VERS3DS:
      begin
        stream.Read(FVersion, 4);
        if not (FVersion = 3) then
        begin
          stream.Seek(0,3);
          //???????
        end;
      end;
      EDIT3DS:
      begin
        //No nothing seems to be needed here...
      end;

      MASTER_SCALE:
      begin
        stream.Read(tempscale, sizeof(tempscale));
        fmasterscale:=tempscale;
      end;

      //read matrial data
      MAT_MATRIAL:
      begin
        //inc(mcount);
        //setlength(HChunk.Material,mcount);
        //moved to MAT_MATNAME (thanks to non-standard 3ds files)
      end;
      MAT_MATNAME:
      begin
        inc(mcount);
        setlength(FMaterial, mcount);
        FNumMaterials := FNumMaterials + 1;
        FMaterial[mcount - 1] := FMaterialClass.Create(self);
        FMaterial[mcount - 1].Name := '';
        FMaterial[mcount - 1].Transparency := 1.0;
        FMaterial[mcount - 1].HasTextureMap := false;
        FMaterial[mcount - 1].HasBumpMap := false;
        FMaterial[mcount - 1].HasOpacMap := false;
        chrbuf := 1;
        while chrbuf <> 0 do
        begin
          stream.Read(chrbuf, 1);
          FMaterial[mcount - 1].Name :=
            FMaterial[mcount - 1].Name + chr(chrbuf);
        end;
        stringbuffer := '';
        stringbuffer := string(FMaterial[mcount - 1].Name);
        //TODO: rewrite
        Delete(stringbuffer, length(stringbuffer), 1);
        FMaterial[mcount - 1].Name := StringBuffer;

//        Log.Writeln(FMaterial[mcount - 1].FName);
        FMaterial[mcount-1].TwoSided := False;
      end;
      MAT_AMBIENT:
      begin
          FMaterial[mcount - 1].IsAmbient := True;
          FMaterial[mcount - 1].IsDiffuse := False;
          FMaterial[mcount - 1].IsSpecular := False;
          FMaterial[mcount - 1].hasmaterial := True;
      end;
      MAT_DIFFUSE:
      begin
          FMaterial[mcount - 1].IsAmbient := False;
          FMaterial[mcount - 1].IsDiffuse := True;
          FMaterial[mcount - 1].IsSpecular := False;
          FMaterial[mcount - 1].hasmaterial := True;
      end;
      MAT_SPECULAR:
      begin
          FMaterial[mcount - 1].IsAmbient := False;
          FMaterial[mcount - 1].IsDiffuse := False;
          FMaterial[mcount - 1].IsSpecular := True;
          FMaterial[mcount - 1].hasmaterial := True;
      end;
      MAT_TEXTURE:
      begin
        bumpmap:=false;
          FMaterial[mcount - 1].HasTexturemap := True;
          FMaterial[mcount - 1].Us := 0;
          FMaterial[mcount - 1].Vs := 0;
          FMaterial[mcount - 1].Uoff := 0;
          FMaterial[mcount - 1].Voff := 0;
      end;
      MAT_BUMPMAP:
      begin
        bumpmap:=true;
        bump := 100;
        stream.Read(dummy, sizeof(TChunkHdr));
        //percentage chunk header overslaan...
        stream.Read(bump, 2);
        FMaterial[mcount - 1].BumpmapStrength := bump / 10000.0;
      end;
      MAT_OPACMAP:
      begin
        opacmap:=true;
        opac := 100;
        stream.Read(dummy, sizeof(TChunkHdr));
        //percentage chunk header overslaan...
        stream.Read(opac, 2);
        //FMaterial[mcount - 1].FOpacmapStrength := opac / 10000.0;
      end;
      MAT_MAPFILE:
      begin
        chrbuf := 1;
        mapname:='';
        while chrbuf <> 0 do
        begin
          stream.Read(chrbuf, 1);
          MapName := MapName + chr(chrbuf);
        end;
        Delete(MapName, length(MapName), 1);

        //Texture
        If (Not BumpMap) and (Not OpacMap) then
        begin
          //texture filename
          FMaterial[mcount - 1].FileName := mapname;
        end;

        //Bumpmap
        if BumpMap then
        begin
          //bumpmap filename
          FMaterial[mcount - 1].BumpMapFileName := mapname;
          FMaterial[mcount - 1].HasBumpMap := true;
          bumpmap:=false;
        end;

        //Opacmap
        if OpacMap then
        begin
          //opacmap filename
          FMaterial[mcount - 1].OpacMapFileName := mapname;
          FMaterial[mcount - 1].HasOpacMap := true;
          opacmap:=false;
        end;

      end;
      MAT_VSCALE:
      begin
      singlebuffer:=0.0;
      stream.Read(singlebuffer, 4);
      FMaterial[mcount - 1].Vs := singlebuffer;
      end;
      MAT_USCALE:
      begin
      singlebuffer:=0.0;
      stream.Read(singlebuffer, 4);
      FMaterial[mcount - 1].Us := singlebuffer;
      end;
      MAT_VOFF:
      begin
      singlebuffer:=0.0;
      stream.Read(singlebuffer, 4);
      FMaterial[mcount - 1].Voff := singlebuffer;
      end;
      MAT_UOFF:
      begin
      singlebuffer:=0.0;
      stream.Read(singlebuffer, 4);
      FMaterial[mcount - 1].Uoff := singlebuffer
      end;
      MAT_TEXROT:
      begin
      singlebuffer:=0.0;
      stream.Read(singlebuffer, 4);
      FMaterial[mcount - 1].Rotate := singlebuffer;
      end;
      MAT_COLOR24:
      begin
        rgbcolor.r :=0;
        rgbcolor.b :=0;
        rgbcolor.g :=0;
        stream.Read(rgbcolor, sizeof(rgbcolor));
        if FMaterial[mcount - 1].IsDiffuse then
        begin
            FMaterial[mcount - 1].DiffuseRed := rgbcolor.r / 255;
            FMaterial[mcount - 1].DiffuseGreen := rgbcolor.g / 255;
            FMaterial[mcount - 1].DiffuseBlue := rgbcolor.b / 255;
        end;
        if FMaterial[mcount - 1].IsAmbient then
        begin
            FMaterial[mcount - 1].AmbientRed := rgbcolor.r / 255;
            FMaterial[mcount - 1].AmbientGreen := rgbcolor.g / 255;
            FMaterial[mcount - 1].AmbientBlue := rgbcolor.b / 255;
        end;
        if FMaterial[mcount - 1].IsSpecular then
        begin
            FMaterial[mcount - 1].SpecularRed := rgbcolor.r / 255;
            FMaterial[mcount - 1].SpecularGreen := rgbcolor.g / 255;
            FMaterial[mcount - 1].SpecularBlue := rgbcolor.b / 255;
        end;
      end;
      MAT_COLOR:
      begin
        color.r := 0.0;
        color.g := 0.0;
        color.b := 0.0;

        stream.Read(color, sizeof(t3dscolor));
        if FMaterial[mcount - 1].IsDiffuse then
        begin
            FMaterial[mcount - 1].DiffuseRed := color.r;
            FMaterial[mcount - 1].DiffuseGreen := color.g;
            FMaterial[mcount - 1].DiffuseBlue := color.b;
        end;
        if FMaterial[mcount - 1].IsAmbient then
        begin
            FMaterial[mcount - 1].AmbientRed := color.r;
            FMaterial[mcount - 1].AmbientGreen := color.g;
            FMaterial[mcount - 1].AmbientBlue := color.b;
        end;
        if FMaterial[mcount - 1].IsSpecular then
        begin
            FMaterial[mcount - 1].SpecularRed := color.r;
            FMaterial[mcount - 1].SpecularGreen := color.g;
            FMaterial[mcount - 1].SpecularBlue := color.b;
        end;
      end;
      MAT_TRANSPARENCY:
      begin
        transp := 100;
        stream.Read(dummy, sizeof(TChunkHdr));
        //percentage chunk header overslaan...
        stream.Read(transp, 2);
        FMaterial[mcount - 1].Transparency := 1.0 - transp / 100.0;
      end;
      MAT_TWO_SIDE :
      begin
        // This chunk contains nothing but the header. If it's present,
        // the current material is two-sided and won't get backface-culled
        FMaterial[mcount-1].TwoSided := True;
      end;

      //read submeshes (objects) ...
      EDIT_OBJECT{, OBJ_HIDDEN}:
      begin
        inc(acount);
        setlength(FMesh, acount);
        setlength(FRenderOrder, acount);
        chrbuf := 1;
        FMesh[acount - 1] := FMeshClass.Create(self); //create
        FMesh[acount - 1].Visible := True;
        FMesh[acount - 1].Name := '';
        Fmesh[acount - 1].Id:=acount; //store an id per mesh...
        FRenderOrder[acount - 1] := acount - 1;
        while chrbuf <> 0 do
        begin
          stream.Read(chrbuf, 1);
          FMesh[acount - 1].Name := FMesh[acount - 1].Name + chr(chrbuf);
        end;

        stringbuffer:=FMesh[acount - 1].Name;
        Delete(stringbuffer, length(stringbuffer), 1);
        FMesh[acount - 1].Name := stringbuffer;

        //TODO: add procedure to mesh to add material
        //set dummy material
        //setlength(FMesh[acount - 1].MatName, 1);
        FMesh[acount - 1].MatName[0] := ''; //???

      end;
      OBJ_TRIMESH:
      begin
        matarr := 0; //reset matarr to 0 for every submesh
        FNumMeshes := FNumMeshes + 1;
      end;
      OBJ_LIGHT:
      begin
        //do nothing yet skip it
        Stream.Seek(Buffer.ChunkLE - sizeof(TChunkHdr), soFromCurrent);
        FNumMeshes := FNumMeshes + 1;
      end;
      OBJ_CAMERA:
      begin
        //do nothing yet skip it
        Stream.Seek(Buffer.ChunkLE - sizeof(TChunkHdr), soFromCurrent);
        FNumMeshes := FNumMeshes + 1;
      end;

      //ReadVertices...
      TRI_VERTEXL:
      begin
        stream.Read(Count, 2);

        if Count > 0 then
        begin
          FMesh[acount - 1].NumVertex := Count;

          for i := 0 to Count - 1 do
          begin
            stream.Read(tv, 12);
            FMesh[acount - 1].Vertex[i] := tv;
            FMesh[acount - 1].BoneId[i] := -1;

            //Set dummy values for meshes without mapping
            tempmap.tu := 0;
            tempmap.tv := 0;
            FMesh[acount - 1].Mapping[i] := tempmap;
          end;
        end
        else
        begin
          FMesh[acount - 1].NumVertex := 0;
        end;
      end;
      TRI_MAPPINGCOORDS:
      begin
        stream.Read(Count, 2);
        //For meshes with texture coord load them from 3ds
        FMesh[acount - 1].NumMappings:=Count;
        for i := 0 to Count - 1 do
        begin
          stream.Read(tm, 8);
          FMesh[acount - 1].Mapping[i] := tm;
        end;
      end;
      TRI_MATERIAL:
      begin
        chrbuf := 1;
        inc(matarr);
        //TODO: reimplement setnumber of materials...
        //setlength(FMesh[acount - 1].MatName, matarr);
        FMesh[acount - 1].MatName[matarr - 1] := '';
        while chrbuf <> 0 do
        begin
          stream.Read(chrbuf, 1);
          FMesh[acount - 1].MatName[matarr - 1] :=
            FMesh[acount - 1].MatName[matarr - 1] + chr(chrbuf);
        end;
        StringBuffer:=string(FMesh[acount - 1].MatName[matarr - 1]);
        Delete(StringBuffer, length(StringBuffer), 1);
        FMesh[acount - 1].MatName[matarr - 1]:='';
        FMesh[acount - 1].MatName[matarr - 1]:=StringBuffer;

        //look up and set matid for vertices with this material
        stream.Read(matcount, 2);
        if matcount > 0 then //hmm matcount should be higher then 0????
        begin
          //TODO: Rewrite Reimplement Number of MatId.
          //SetLength(FMesh[acount - 1].MatId,
          //  FMesh[acount - 1].NumFaces div 3);
          for matcountloop := 0 to matcount - 1 do
          begin
            stream.Read(mati, 2);
            FMesh[acount - 1].MatId[mati] :=
              GetMaterialIDbyName(FMesh[acount - 1].MatName[matarr - 1]);
          end;
        end;
      end;
      TRI_VISIBLE:
      begin
        //By default all meshes are visible as the only values returned seem to be false
        stream.Read(tempbool, 1);
        FMesh[acount-1].visible:=tempbool;
      end;

      TRI_FACEL1:
      begin
        stream.Read(Count, 2);
        FMesh[acount - 1].NumVertexIndices := Count * 3;
        FMesh[acount - 1].NumNormals := Count * 3;
        FMesh[acount - 1].NumNormalIndices := Count * 3;
        FMesh[acount - 1].NumMappingIndices := Count * 3;
        //FMesh[acount - 1].NumFaceRecords := Count;

        for i := 0 to Count-1 do
        begin
          WordBuffer:=0;
          stream.Read(WordBuffer, 2);
          FMesh[acount - 1].Face[i * 3]:=WordBuffer;

          WordBuffer:=0;
          stream.Read(WordBuffer, 2);
          FMesh[acount - 1].Face[i * 3 + 1] := WordBuffer;

          WordBuffer:=0;
          stream.Read(WordBuffer, 2);
          FMesh[acount - 1].Face[i * 3 + 2]:=WordBuffer;

          wordbuffer:=0;
          stream.Read(WordBuffer, 2);

          //temp_diff:=wordbuffer AND $000F;
          //FMesh[acount - 1].Face[i * 3 + 3]:=(temp_diff AND $0004) OR 2;
          //FMesh[acount - 1].Face[i * 3 + 4]:=(temp_diff AND $0002) OR 1;
          //FMesh[acount - 1].Face[i * 3 + 5]:=(temp_diff AND $0001);

          //TODO: waarom dubbel? Een methode overhouden!!!
          //Alleen TFace overhouden!!!

          //copy vertex to normal data
          FMesh[acount - 1].Normal[i * 3] := FMesh[acount - 1].Face[i * 3];
          FMesh[acount - 1].Normal[i * 3 + 1] := FMesh[acount - 1].Face[i * 3 + 1];
          FMesh[acount - 1].Normal[i * 3 + 2] := FMesh[acount - 1].Face[i * 3 + 2];

          FMesh[acount - 1].Map[i * 3] := FMesh[acount - 1].Face[i * 3];
          FMesh[acount - 1].Map[i * 3 + 1] := FMesh[acount - 1].Face[i * 3 + 1];
          FMesh[acount - 1].Map[i * 3 + 2] := FMesh[acount - 1].Face[i * 3 + 2];

          //FMesh[acount - 1].Normal[i * 3 + 3] :=
          //  FMesh[acount - 1].Face[i * 3 + 3];
          //FMesh[acount - 1].Normal[i * 3 + 4] :=
          //  FMesh[acount - 1].Face[i * 3 + 4];
          //FMesh[acount - 1].Normal[i * 3 + 5] :=
          //  FMesh[acount - 1].Face[i * 3 + 5];



          //copy vertex and normal indexes to face structure
          //TODO FIX ASAP no faces are storedd.....

          (*
          tempface:= FMesh[acount -1].Faces[i];
          tempface.vertex[0]:=FMesh[acount - 1].Face[i * 3];
          tempface.vertex[1]:=FMesh[acount - 1].Face[i * 3 + 1];
          tempface.vertex[2]:=FMesh[acount - 1].Face[i * 3 + 2];

          //tempface.vertex[3]:=FMesh[acount - 1].Face[i * 3 + 3];
          //tempface.vertex[4]:=FMesh[acount - 1].Face[i * 3 + 4];
          //tempface.vertex[5]:=FMesh[acount - 1].Face[i * 3 + 5];

          tempface.normal[0]:=FMesh[acount - 1].Normal[i * 3];
          tempface.normal[1]:=FMesh[acount - 1].Normal[i * 3 + 1];
          tempface.normal[2]:=FMesh[acount - 1].Normal[i * 3 + 2];

          //tempface.normal[3]:=FMesh[acount - 1].Normal[i * 3 + 3];
          //tempface.normal[4]:=FMesh[acount - 1].Normal[i * 3 + 4];
          //tempface.normal[5]:=FMesh[acount - 1].Normal[i * 3 + 5];

          FMesh[acount -1].Faces[i]:=tempface;
          *)
          
        end;
      end;

      TRI_MATRIX:
      begin
          //Chunk.Data.MeshMatrix := AllocMem(SizeOf(TMeshMatrix));

          for i := 0 to 11 do
          begin
          WordBuffer:=0;
          stream.Read(SingleBuffer, sizeof(singlebuffer));
          FMesh[acount -1].Matrix[i] := SingleBuffer;
          end;
          FMesh[acount -1].Matrix[12] := 0;
          FMesh[acount -1].Matrix[13] := 0;
          FMesh[acount -1].Matrix[14] := 0;
          FMesh[acount -1].Matrix[15] := 1;

      end;

      // read in keyframe data if available (for now only for pivot)
      KEYF3DS:
      begin
        //No nothing seems to be needed here...
      end;
      KEYF_OBJDES:
      begin
        //if all is ok then for every submesh a keyf_objdes exists
        inc(keyfcount);
      end;
      KEYF_OBJPIVOT:
      begin
        //read in pivot point
        stream.Read(tv, 12);
        if keyfcount <= FNumMeshes then
          //stupid way to do, but some 3ds files have less meshes then 'bones'
          FMesh[keyfcount - 1].Pivot := tv;
      end;
      else
        stream.Seek(buffer.chunkLE - sizeof(TChunkHdr), soFromCurrent);
    end;
  end;

  //TODO: implement missing procedures and functions below
  //calculate some things afterwards...
  movetopivot;                   //if pivots are used in 3ds move submeshes

//  fmasterscale:=fmasterscale/100; //?? Am i allowed to do this?

  CalculateScale;                //calculate new vertexes with masterscale
  calcvnormals;                  //calculate vertex normals for a smooth looking 3ds
  CalculateSize;                 //calculate min and max size
  CalculateRenderOrder;          //transparent items should be rendered last

  //preload textures...
  if FNumMaterials > 0 then
  for m := 0 to FNumMaterials - 1 do
  begin
    //TMaterial(FMaterial[m]).updatetexture; //TODO: weer aanzetten.
    //Initial texture loading should be independent of opengl/dx
  end;

  //set id's
  for m := 0 to FNumMeshes - 1 do
  begin
    FMesh[m].id:=m+1;
  end;

  //set connectivity of faces
  for m := 0 to FNumMeshes - 1 do
  begin
    //if FMesh[m].FNumIndices > 0 then FMesh[m].SetConnectivity;
  end;

  //calculate planes for faces
  for m := 0 to FNumMeshes - 1 do
  begin
    //if FMesh[m].FNumIndices > 0 then FMesh[m].CalculatePlanes;
  end;

//  log.Writeln('done loading 3ds');

end;

initialization
//TBaseModel.RegisterFileFormat('3ds', '3d studio binary model', T3dsModel);
RegisterModelFormat('3ds', '3d studio binary model', T3dsModel);

finalization
//TBaseModel.UnRegisterModelClass(T3dsModel);
UnregisterModelClass(T3dsModel);

end.
